<?php

namespace NioModules\WdBank\Controllers;

use App\Helpers\NioHash;
use Illuminate\Http\Request;
use NioModules\WdBank\WdBankModule;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use NioModules\WdBank\Services\UserAccountService;

class UserAccountController extends Controller
{
    private $accountService;

    public function __construct(UserAccountService $accountService)
    {
        $this->accountService = $accountService;
    }
    /**
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\View\View
     * @version 1.0.0
     * @since 1.0
     */
    public function form(Request $request)
    {
        if ($request->ajax()) {
            return view('WdBank::account-form', $this->accountService->form($request));
        }

        return "";
    }

    public function save(Request $request)
    {
        $input = $this->accountService->getFormData($request);
        
        try {
            $this->accountService->save($input);

            if ($request->get('quick_added') == 'yes') {
                return redirect()->route('withdraw.redirect.amount');
            }

            return response()->json(["msg" => __('Your bank account successfully added.'), "msg_title" => __('Account Added')]);
        } catch (ValidationException $e) {
            throw $e;
        }

        return response()->json(["type" => $input->data['type'], "message" => $input->data['message'], "msg" => $input->data['errors']]);
    }

    public function edit($hash)
    {
        $id = NioHash::toID($hash);
        $method = $this->accountService->getWithdrawMethod();
        $currencies = $method->currencies;
        $fallback = isset($method->currencies[0]) ? $method->currencies[0] : '';
        $default = data_get($method->config, 'meta.currency', $fallback);

        $userAccount = $this->accountService->userAccount($hash);

        if (blank($userAccount)) {
            throw ValidationException::withMessages(['acc' => [0 => __('Invalid Account'), 1 => __('Sorry, account may invalid or not found.')]]);
        }

        $countries = filtered_countries($method);
        $action = route('user.withdraw.account.' . WdBankModule::SLUG . '.update', ['id' => NioHash::of($id)]);
        $quickAdd = false;

        return view('WdBank::account-form', compact('userAccount', 'method', 'action', 'currencies', 'default', 'countries', 'quickAdd'));
    }

    public function update($hash)
    {
        $request = request();
        $input = $this->accountService->getFormData($request);

        try {
            $this->accountService->update($hash, $input);
            return response()->json(["msg" => __('The bank account successfully updated.'), "msg_title" => __('Account Updated')]);
        } catch (ValidationException $e) {
            throw $e;
        }

        return response()->json(["type" => $input->data['type'], "message" => $input->data['message'], "msg" => $input->data['errors']]);
    }

    public function delete($hash)
    {
        $this->accountService->delete($hash);
        return response()->json(['msg' => __('The account successfully deleted.'), "msg_title" => __('Account Deleted')]);
    }
}
